/* Copyright (c) 1997, 2009, Oracle and/or its affiliates. 
All rights reserved. */

/*

   NAME
     imgdemo.c - out-of-the-box demo using Oracle Multimedia ORDImage object.

   DESCRIPTION
      This program demonstrates how to use the Oracle Multimedia ORDImage
      object to load an image into the database, alter its attributes,
      and then write the data to a file for viewing.  The demo will use an
      image data file provided with Multimedia named imgdemo.bin
      as its input so no parameter is required.

      The process method will be used to scale and crop the original
      image so expect the resultant image in imgdemo.out to reflect
      this change.

      Sample usage:    imgdemo <optional-image-filename>
   
      imgdemo.bin    --  image to be manipulated
      imgdemo.out    --  resultant image filename

      Optionally, a user provided image file can be specified on
      the command line as shown above, provided that it resides
      in the IMGDEMODIR directory (see assumptions below).

   ENVIRONMENT:

      The following assumptions are made:

       1) Oracle Multimedia has been installed and PUBLIC
          has EXECUTE privilege on Multimedia data types.

       2) A demo user has been created and been granted create 
          table, create any directory, and drop any directory 
          privileges.
       
       3) setup.sql script has been run to create imgdemodir 
          directory object. 

       4) imgdemo.bin is a valid image file that resides in the
          IMGDEMODIR directory and the user has read/write access
          to the directory.

   PUBLIC FUNCTION(S)

   PRIVATE FUNCTION(S)

   RETURNS

   NOTES
      After demo has been executed, use a rendering tool such as xv
      to view both the original and the modified images.  The table
      IMGDEMOTAB is left in the schema account in the default database
      for the user's viewing as well.  The IMGDEMODIR directory will 
      likewise be left in the system account.
*/


#include <stdio.h>
#include <string.h>

#ifdef _WIN32
/*For Windows platform */
#include <WTypes.h>
#include <Windows.h>
static DWORD oldMode;
#else
/*For Linux/Unix platform */
#include <termios.h>
static struct termios oldMode;
#endif

#ifndef OCI_ORACLE
#include <oci.h>
#endif

/* local routines */
static void setEchoOn(void);
static void setEchoOff(void);
static sb4 init_handles();
static sb4 attach_server();
static sb4 log_on();
static void logout();
static sb4 create_table();
static void drop_table();
static sb4 fetch_blob();
static sb4 load_cart();
static sb4 mod_img();
static sb4 blob_to_file();
static void report_error();

#define TRUE       1
#define FALSE      0

#define MAXBUFLEN  16384
#define STMTLEN    512
#define FNAMELEN   80
#define BUF_LEN    31

static OCIEnv        *envhp;
static OCIServer     *srvhp;
static OCISvcCtx     *svchp;
static OCIError      *errhp;
static OCISession    *authp;
static OCIStmt       *stmthp;
static OCILobLocator *blob, *bfile;
static OCIDefine     *defnp1;

static char *fname = "imgdemo.out";

/*database username and password*/
static char user[BUF_LEN];
static char upwd[BUF_LEN];

int main(int argc, char *argv[])
{
  char iname[FNAMELEN];

  /* read the image file name if provided, otherwise use default */
  if (argc == 2){
    strncpy(iname, argv[1], FNAMELEN);
  }
  else{
    strncpy(iname, "imgdemo.bin", FNAMELEN);
  }

  /* prompt database username */
  (void) fprintf(stdout, "Username: ");
  fgets(user, BUF_LEN, stdin);   
  user[strlen(user)-1] = 0; /* remove carriage return */

  /* prompt database password */
  setEchoOff();
  (void) fprintf(stdout, "Password: ");
  fgets(upwd, BUF_LEN, stdin);  
  upwd[strlen(upwd)-1] = 0; /* remove carriage return */
  setEchoOn();

  if (init_handles())
  {
    (void) fprintf(stdout,"FAILED: init_handles()\n");
    return OCI_ERROR;
  }

  if (attach_server())
  {
    (void)fprintf (stdout, "FAILED: attach_server()\n");
    logout();
    return OCI_ERROR;
  }

  if (log_on(user,upwd))
  {
    (void) fprintf(stdout,"FAILED: log_on()\n");
    logout();
    return OCI_ERROR;
  }

  /* zero out the password for security reason */
  memset(upwd, '\0', BUF_LEN);

  if (create_table(iname))
  {
    (void) fprintf(stdout,"FAILED: create_table()\n");
    logout();
    return OCI_ERROR;
  }

  (void)fprintf (stdout, "\nLoading data into cartridge...\n");
  if (load_cart())
  {
    (void) fprintf(stdout,"FAILED: load_cart()\n");
    logout();
    return OCI_ERROR;
  }

  (void)fprintf (stdout, "\nModifying image characteristics...\n");
  if (mod_img())
  {
    (void)fprintf (stdout, "\nFAILED: mod_img()\n");
    logout();
    return OCI_ERROR;
  }

  (void)fprintf (stdout, "\nWriting image to file %s...\n",fname);
  if (blob_to_file())
  {
    (void)fprintf (stdout, "\nFAILED: blob_to_file()\n");
    logout();
    return OCI_ERROR;
  }

  (void)fprintf (stdout, "\nDisconnecting from database...\n");
  logout();

  (void)fprintf (stdout, "\nDemo completed successfully.\n");
  return OCI_SUCCESS;

}  /* end main */


#ifdef _WIN32
/* Functions for Windows */

/* Function on Windows to set the command line input echo off*/
void setEchoOff(void){
  DWORD newMode = 0;
  HANDLE hStdin = GetStdHandle(STD_INPUT_HANDLE); 
  GetConsoleMode(hStdin, &oldMode);
  newMode = oldMode;
  SetConsoleMode(hStdin, newMode & (~ENABLE_ECHO_INPUT)); 
  return;
}

/* Function on Windows to set the command line input echo on*/
void setEchoOn(void){
    HANDLE hStdin = GetStdHandle(STD_INPUT_HANDLE); 
    SetConsoleMode(hStdin, oldMode);
}

#else 
/* Functions for Unix */

/* Function on Unix to set the command line input echo off*/
void setEchoOff(void)
{
  struct termios newMode;
  tcgetattr(0,&oldMode);
  newMode = oldMode;
  newMode.c_lflag &= (~ECHO);
  tcsetattr(0,TCSANOW,&newMode);
  return;
}

/* Function on Unix to set the command line input echo on*/
void setEchoOn(void)
{
  tcsetattr(0,TCSANOW,&oldMode);
  return;
}
#endif


/* 
 * -----------------------------------------------------------------
 * init_handles - initialize environment, allocate handles, etc.
 * -----------------------------------------------------------------
 */

sb4 init_handles()
{
  sword status;

  if (OCIInitialize((ub4) OCI_DEFAULT, 
                    (dvoid *)0, 
                    (dvoid * (*)(dvoid *, size_t)) 0,
                    (dvoid * (*)(dvoid *, dvoid *, size_t))0,
                    (void (*)(dvoid *, dvoid *)) 0 ))
  {
    (void) fprintf(stdout,"FAILED: OCIInitialize()\n");
    return OCI_ERROR;
  }
 
  /* initialize environment handle */
  if (OCIEnvInit((OCIEnv **) &envhp, 
                 (ub4) OCI_DEFAULT,
                 (size_t) 0, 
                 (dvoid **) 0 ))
  {
    (void) fprintf(stdout,"FAILED: OCIEnvInit()\n");
    return OCI_ERROR;
  }
 
  if (OCIHandleAlloc((dvoid *) envhp, 
                     (dvoid **) &svchp,
                     (ub4) OCI_HTYPE_SVCCTX, 
                     (size_t) 0, 
                     (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIHandleAlloc()\n");
    return OCI_ERROR;
  }

  if (OCIHandleAlloc((dvoid *) envhp, 
                     (dvoid **) &errhp,
                     (ub4) OCI_HTYPE_ERROR, 
                     (size_t) 0, 
                     (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIHandleAlloc()\n");
    return OCI_ERROR;
  }
 
  if (OCIHandleAlloc((dvoid *) envhp, 
                     (dvoid **) &stmthp,
                     (ub4) OCI_HTYPE_STMT,  
                     (size_t) 0, 
                     (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIHandleAlloc()\n");
    return OCI_ERROR;
  }
 
  if (OCIHandleAlloc((dvoid *) envhp, 
                     (dvoid **) &srvhp,
                     (ub4) OCI_HTYPE_SERVER, 
                     (size_t) 0, 
                     (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIHandleAlloc()\n");
    return OCI_ERROR;
  }

  if (OCIHandleAlloc((dvoid *) envhp, 
                     (dvoid **) &authp,
                     (ub4) OCI_HTYPE_SESSION, 
                     (size_t) 0, 
                     (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIHandleAlloc()\n");
    return OCI_ERROR;
  }

  /* allocate the lob locator variables */
  if (OCIDescriptorAlloc((dvoid *) envhp, 
                         (dvoid **) &blob, 
                         (ub4)OCI_DTYPE_LOB, 
                         (size_t) 0, 
                         (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIDescriptorAlloc(blob)\n");
    return OCI_ERROR;
  }

  /* allocate the lob locator variables - will change to OCI_DTYPE_FILE */
  if (OCIDescriptorAlloc((dvoid *) envhp, 
                         (dvoid **) &bfile,
                         (ub4)OCI_DTYPE_LOB, 
                         (size_t) 0,  
                         (dvoid **) 0))
  {
    (void) fprintf(stdout,"FAILED: OCIDescriptorAlloc(bfile)\n");
    return OCI_ERROR;
  }

  return OCI_SUCCESS;

}  /* end init_handles */


/*
 * -----------------------------------------------------------------
 * attach_server - attach to default server
 * -----------------------------------------------------------------
 */
sb4 attach_server()
{
  /* attach to the server - use default host */
  if (OCIServerAttach(srvhp, 
                      errhp, 
                      (text *) 0,
                      (sb4) 0, 
                      (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIServerAttach()\n");
    return OCI_ERROR;
  }

  /* set the server attribute in the service context */
  if (OCIAttrSet((dvoid *) svchp, 
                 (ub4) OCI_HTYPE_SVCCTX,
                 (dvoid *) srvhp, 
                 (ub4) 0, 
                 (ub4) OCI_ATTR_SERVER, 
                 errhp))
  {
    (void) fprintf(stdout,"FAILED: OCIAttrSet()\n");
    return OCI_ERROR;
  }
 
  return (OCI_SUCCESS);
 
}  /* end attach_server */


/* 
 * -----------------------------------------------------------------
 * log_on - log on to server
 * -----------------------------------------------------------------
 */
sb4 log_on(
  char *uid,
  char *pwd
)
{
  if (OCIAttrSet((dvoid *) authp, 
                 (ub4) OCI_HTYPE_SESSION,
                 (dvoid *) uid, 
                 (ub4) strlen((char *)uid),
                 (ub4) OCI_ATTR_USERNAME, 
                 errhp))
  {
    (void) fprintf(stdout,"FAILED: OCIAttrSet()\n");
    return OCI_ERROR;
  }
 
  if (OCIAttrSet((dvoid *) authp, 
                 (ub4) OCI_HTYPE_SESSION,
                 (dvoid *) pwd, 
                 (ub4) strlen((char *)pwd),
                 (ub4) OCI_ATTR_PASSWORD, 
                 errhp))
  {
    (void) fprintf(stdout,"FAILED: OCIAttrSet()\n");
    return OCI_ERROR;
  }

  /* log on */
  if (OCISessionBegin(svchp,  
                      errhp, 
                      authp, 
                      (ub4) OCI_CRED_RDBMS,
                      (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCISessionBegin()\n");
    return OCI_ERROR;
  }
 
  /* set the session attribute in the service context */
  if (OCIAttrSet((dvoid *) svchp, 
                 (ub4) OCI_HTYPE_SVCCTX, 
                 (dvoid *) authp,
                 (ub4) 0,  
                 (ub4) OCI_ATTR_SESSION, errhp))
  {
    (void) fprintf(stdout,"FAILED: OCIAttrSet()\n");
    return OCI_ERROR;
  }
 
  return OCI_SUCCESS;

}  /* end log_on */


/*
 * -----------------------------------------------------------------
 * create_table - Create table IMGDEMOTAB and insert one row.
 * -----------------------------------------------------------------
 */

sb4 create_table(char *iname)
{
  text *crtstmt = (text *) 
"CREATE TABLE IMGDEMOTAB (C1 INT, FILEIMAGE ORDSYS.ORDImage, BLOBIMAGE ORDSYS.ORDImage)";
  text insstmt[STMTLEN]; 

  sprintf ((char*)insstmt,"INSERT INTO IMGDEMOTAB VALUES (1,ORDSYS.ORDImage.init('file', 'IMGDEMODIR','%s'),ORDSYS.ORDImage.init())",iname);

  /*
   * drop table first if it exists and then re-create.
   */
  drop_table();

  /*
   * create table
   */
  if (OCIStmtPrepare(stmthp, 
                     errhp, 
                     crtstmt, 
                     (ub4) strlen((char *) crtstmt),
                     (ub4) OCI_NTV_SYNTAX, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtPrepare() crtstmt\n");
    report_error();
    return OCI_ERROR;
  }

  (void)fprintf (stdout, "\nCreating and populating table IMGDEMOTAB...\n");
  if (OCIStmtExecute(svchp, 
                     stmthp, 
                     errhp, 
                     (ub4) 1, 
                     (ub4) 0,
                     (CONST OCISnapshot *) 0, 
                     (OCISnapshot *) 0, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: creating table\n");
    report_error();
    return OCI_ERROR;
  }

  /*
   * populate table with simple insert
   */
  if (OCIStmtPrepare(stmthp, 
                     errhp, 
                     insstmt, 
                     (ub4) strlen((char *) insstmt),
                     (ub4) OCI_NTV_SYNTAX, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtPrepare() insstmt\n");
    report_error();
    return OCI_ERROR;
  }

  if (OCIStmtExecute(svchp, 
                     stmthp, 
                     errhp, 
                     (ub4) 1, 
                     (ub4) 0,
                     (CONST OCISnapshot *) 0, 
                     (OCISnapshot *) 0, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtExecute() insstmt\n");
    report_error();
    return OCI_ERROR;
  }

  (void) OCITransCommit(svchp, errhp, (ub4)0);

  return OCI_SUCCESS;

}  /* end create_table */



/*
 *---------------------------------------------------------------------
 * fetch_blob - fetch the blob locator
 *---------------------------------------------------------------------
 */

sb4 fetch_blob()
{
  text  *sqlstmt = (text *)"SELECT A.BLOBIMAGE.getContent() FROM IMGDEMOTAB A";

  if (OCIStmtPrepare(stmthp, 
                     errhp, 
                     sqlstmt, 
                     (ub4) strlen((char *)sqlstmt),
                     (ub4) OCI_NTV_SYNTAX, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtPrepare() sqlstmt\n");
    report_error();
    return OCI_ERROR;
  }
  
  /*
   * define the output blob variable by position 
   */
  if (OCIDefineByPos
             (stmthp, 
              &defnp1, 
              errhp, 
              (ub4) 1,
              (dvoid *) &blob, 
              (sb4) -1,
              (ub2) SQLT_BLOB,
              (dvoid *) 0, 
              (ub2 *) 0,
              (ub2 *) 0, 
              (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIDefineByPos()\n");
    report_error();
    return OCI_ERROR;
  }

  /* execute the select and fetch one row */
  if (OCIStmtExecute(svchp, 
                     stmthp, 
                     errhp, 
                     (ub4) 1, 
                     (ub4) 0,
                     (CONST OCISnapshot*) 0, 
                     (OCISnapshot*) 0,  
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtExecute() sqlstmt\n");
    report_error();
    return OCI_ERROR;
  }

  return OCI_SUCCESS;

}  /* fetch_blob */


/*
 *-------------------------------------------------------------------
 * drop_table - Drop table IMGDEMOTAB 
 *-------------------------------------------------------------------
 */

void drop_table()
{
  text  *sqlstmt = (text *) "DROP TABLE IMGDEMOTAB";
  ub1   ebuf[256];
  text  *sqlstate=0;
  ub4   errcodep;
 
  (void)fprintf(stdout,"\nDropping table IMGDEMOTAB...\n");
  if (OCIStmtPrepare(stmthp, 
                     errhp, 
                     sqlstmt, 
                     (ub4) strlen((char *) sqlstmt),
                     (ub4) OCI_NTV_SYNTAX, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: drop table\n");
    return;
  }
 
  (void)OCIStmtExecute(svchp, 
                       stmthp, 
                       errhp, 
                       (ub4) 1, 
                       (ub4) 0,
                       (CONST OCISnapshot *) 0, 
                       (OCISnapshot *) 0,
                       (ub4) OCI_DEFAULT);
  return;

}  /* end drop_table */

/*
 * -------------------------------------------------------------------
 * logout - Logoff and disconnect from the server.  Free handles.
 * -------------------------------------------------------------------
 */ 

void logout()
{
  (void) OCISessionEnd(svchp, errhp, authp, (ub4) 0);
  (void) OCIServerDetach(srvhp, errhp, (ub4) OCI_DEFAULT);
 
  (void) fprintf(stdout,"\nLogged off and detached from server.\n");
 
  (void) OCIHandleFree((dvoid *) srvhp, (ub4) OCI_HTYPE_SERVER);
  (void) OCIHandleFree((dvoid *) svchp, (ub4) OCI_HTYPE_SVCCTX);
  (void) OCIHandleFree((dvoid *) errhp, (ub4) OCI_HTYPE_ERROR);
  (void) OCIHandleFree((dvoid *) authp, (ub4) OCI_HTYPE_SESSION);
  (void) OCIDescriptorFree((dvoid *) blob, (ub4) OCI_DTYPE_LOB);
  (void) OCIDescriptorFree((dvoid *) bfile, (ub4) OCI_DTYPE_LOB);
  (void) OCIHandleFree((dvoid *) stmthp, (ub4) OCI_HTYPE_STMT);

  return;

}  /* end logout */


/* 
 * -----------------------------------------------------------------
 * report_error - retrieve error message and print it out.
 * ----------------------------------------------------------------- 
 */
void report_error()
{
  text  msgbuf[1024];
  sb4   errcode;

  (void) OCIErrorGet((dvoid *) errhp, 
                     (ub4) 1, 
                     (text *) NULL, 
                     &errcode,
                     msgbuf, 
                     (ub4) sizeof(msgbuf), 
                     (ub4) OCI_HTYPE_ERROR);
  (void) fprintf(stdout,"ERROR CODE = %d\n", errcode);
  (void) fprintf(stdout,"%s\n", msgbuf);
  return;

}  /* end report_error */


/*
 * -----------------------------------------------------------------
 * load_cart - load data into the image data cartridge
 *
 *   Populate the cartidge columns based on the contents of the bfile.
 *   The sequence of steps is to select the types and then use the 
 *   contents to set the properties and perform the copy from the
 *   bfile to the blob.
 * -----------------------------------------------------------------
 */
sb4 load_cart()
{
text *sqlstmt = (text *)
"DECLARE \
  A ORDSYS.ORDImage;\
  B ORDSYS.ORDImage;\
BEGIN\
 SELECT FILEIMAGE, BLOBIMAGE INTO A,B FROM IMGDEMOTAB FOR UPDATE;\
\
 A.setProperties;\
 A.processCopy('fileFormat=JFIF',B);\
 B.setProperties;\
 UPDATE IMGDEMOTAB SET FILEIMAGE = A;\
 UPDATE IMGDEMOTAB SET BLOBIMAGE = B;\
 COMMIT;\
END;";

  if (OCIStmtPrepare(stmthp, 
                     errhp, 
                     sqlstmt, 
                     (ub4) strlen((char *)sqlstmt),
                     (ub4) OCI_NTV_SYNTAX, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtPrepare() sqlstmt\n");
    report_error();
    return OCI_ERROR;
  }
  
  /* execute the select and process one row */
  if (OCIStmtExecute(svchp, 
                     stmthp, 
                     errhp, 
                     (ub4) 1, 
                     (ub4) 0,
                     (CONST OCISnapshot*) 0, 
                     (OCISnapshot*) 0,  
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtExecute() sqlstmt\n");
    report_error();
    return OCI_ERROR;
  }

  return OCI_SUCCESS;

}  /* end load_cart */

/* 
 * -----------------------------------------------------------------
 * mod_img - modify some characteristics of the image
 *
 * This is accomplished by selecting the cartridge type and then
 * invoking the process method with a command.  The command specified
 * below scales the image by a factor of 2 and cuts a 100x100 pixel
 * piece from the original image.
 * ----------------------------------------------------------------- 
 */
sb4 mod_img()
{
text *sqlstmt = (text *)
"DECLARE \
  B ORDSYS.ORDImage;\
  command VARCHAR2(400);\
BEGIN\
 SELECT BLOBIMAGE INTO B FROM IMGDEMOTAB FOR UPDATE;\
\
 command := 'scale=2 cut=100 100 100 100';\
 B.process(command);\
 B.setProperties;\
 UPDATE IMGDEMOTAB SET BLOBIMAGE = B;\
 COMMIT;\
END;";

  if (OCIStmtPrepare(stmthp, 
                     errhp, 
                     sqlstmt, 
                     (ub4) strlen((char *)sqlstmt),
                     (ub4) OCI_NTV_SYNTAX, 
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtPrepare() sqlstmt\n");
    report_error();
    return OCI_ERROR;
  }
  
  /* execute the select and process one row */
  if (OCIStmtExecute(svchp, 
                     stmthp, 
                     errhp, (ub4) 1, 
                     (ub4) 0,
                     (CONST OCISnapshot*) 0, 
                     (OCISnapshot*) 0,  
                     (ub4) OCI_DEFAULT))
  {
    (void) fprintf(stdout,"FAILED: OCIStmtExecute() sqlstmt\n");
    report_error();
    return OCI_ERROR;
  }

  return OCI_SUCCESS;

}  /* end mod_img */

/* 
 * -----------------------------------------------------------------
 * blob_to_file - copy the contents of a blob to a file
 *
 * Read blob using stream mode into local buffers and then write    
 * data to operating system file.                                   
 * -----------------------------------------------------------------
 */
sb4 blob_to_file()
{
  ub4   offset = 1;
  ub4   loblen = 0;
  ub1   bufp[MAXBUFLEN];
  ub4   amtp = 0;
  sword retval;
  ub4   piece = 0;
  ub4   remainder;
  FILE  *fp;
  sb4   status = OCI_SUCCESS;

  /*
   * fetch the blob from which data will be read
   */
  if (fetch_blob())
  {
     (void)fprintf(stdout, "FAILED: fetch_blob()\n");
     return (OCI_ERROR);
  }

  fp = fopen((char *)fname, (const char *) "wb");

  (void) OCILobGetLength(svchp, errhp, blob, &loblen);
  amtp = loblen;

  memset(bufp, '\0', MAXBUFLEN);

  retval = OCILobRead(svchp, 
                      errhp, 
                      blob, 
                      &amtp, 
                      offset, 
                      (dvoid *) bufp,
                      (loblen < MAXBUFLEN ? loblen : MAXBUFLEN), 
                      (dvoid *)0, 
                      (OCICallbackLobRead) 0,
                      (ub2) 0, 
                      (ub1) SQLCS_IMPLICIT);

  switch (retval)
  {
    case OCI_SUCCESS:             /* only one piece */
      (void) fwrite(bufp, loblen, 1, fp);
      break;
    case OCI_ERROR: 
      report_error();
      break;
    case OCI_NEED_DATA:           /* there are 2 or more pieces */

      remainder = loblen;

      (void) fwrite(bufp, MAXBUFLEN, 1, fp); /* a full buffer to write */

      do
      {
        memset(bufp, '\0', MAXBUFLEN);
        amtp = 0;

        remainder -= MAXBUFLEN;

        retval = OCILobRead(svchp, 
                            errhp, 
                            blob, 
                            &amtp, 
                            offset, 
                            (dvoid *) bufp,
                            (ub4) MAXBUFLEN, 
                            (dvoid *)0, 
                            (OCICallbackLobRead) 0,
                            (ub2) 0, 
                            (ub1) SQLCS_IMPLICIT);

        /* the amount read returned is undefined for FIRST, NEXT pieces */

        if (remainder < MAXBUFLEN)     /* last piece not a full buffer piece */
           (void) fwrite(bufp, remainder, 1, fp);
        else
           (void) fwrite(bufp, MAXBUFLEN, 1, fp);

      } while (retval == OCI_NEED_DATA);
      break;
    default:
      (void) fprintf(stdout,"Unexpected ERROR: OCILobRead() LOB.\n");
      break;
  }

  if (ftell(fp) != loblen)
  {
    (void)fprintf(
       stdout,
       "ERROR: resultant file size of %d differs from source lob size of %d\n",
       ftell(fp),loblen);
    status = OCI_ERROR;
  }

  if (fclose(fp))
     return (OCI_ERROR);
  else
     return (status);

}  /* end blob_to_file */


/* end of file imgdemo.c */

